import { Request, Response, NextFunction } from 'express';
import { AppError } from '@/types';

/**
 * Custom error class for operational errors
 */
export class CustomError extends Error implements AppError {
  statusCode: number;
  isOperational: boolean;

  constructor(message: string, statusCode: number = 500) {
    super(message);
    this.statusCode = statusCode;
    this.isOperational = true;

    Error.captureStackTrace(this, this.constructor);
  }
}

/**
 * Global error handling middleware
 */
export const errorHandler = (
  err: Error | AppError,
  req: Request,
  res: Response,
  next: NextFunction
) => {
  let error = { ...err };
  error.message = err.message;

  // Log error for debugging
  console.error('Error:', {
    message: err.message,
    stack: err.stack,
    url: req.url,
    method: req.method,
    timestamp: new Date().toISOString()
  });

  // Handle Supabase errors
  if (err.message.includes('JWT')) {
    const message = 'Invalid token';
    error = new CustomError(message, 401);
  }

  // Handle validation errors
  if (err.message.includes('validation')) {
    const message = 'Validation error';
    error = new CustomError(message, 400);
  }

  // Handle duplicate key errors
  if (err.message.includes('duplicate key')) {
    const message = 'Resource already exists';
    error = new CustomError(message, 409);
  }

  // Default error response
  const statusCode = (error as AppError).statusCode || 500;
  const message = error.message || 'Internal server error';

  res.status(statusCode).json({
    success: false,
    error: message,
    ...(process.env.NODE_ENV === 'development' && { stack: err.stack })
  });
};

/**
 * Not found handler middleware
 */
export const notFoundHandler = (req: Request, res: Response) => {
  res.status(404).json({
    success: false,
    error: `Route ${req.originalUrl} not found`
  });
}; 